/************************************************************************
 * @file: IDBusCommandClient.h
 *
 * @version: 1.1
 *
 * @description: IDBusCommandClient is a common interface class for
 * both receiver and sender command plug-in interface of AM.
 * @component: platform/audiomanager
 *
 * @author: Jens Lorenz, jlorenz@de.adit-jv.com 2016
 *          Mattia Guerra, mguerra@de.adit-jv.com 2016
 *
 * @copyright (c) 2016 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 * @see <related items>
 *
 * @history
 *
 ***********************************************************************/

#ifndef _I_DBUS_COMMAND_CLIENT_H_
#define _I_DBUS_COMMAND_CLIENT_H_

#include <pthread.h>
#include "CAmSerializer.h"
#include "CAmDbusWrapper.h"

#include "CDBusReceiver.h"
#include "IAmCommand.h"

namespace am
{

class CDBusCommandSender;

class IDBusCommandClient
{
public:
    IDBusCommandClient();
    virtual ~IDBusCommandClient();

    /* Receiver methods. Protected ones will be shadowed to ApplicationClient */
protected:
    am_Error_e connect(const am_sourceID_t sourceID, const am_sinkID_t sinkID, am_mainConnectionID_t& mainConnectionID);
    am_Error_e disconnect(const am_mainConnectionID_t mainConnectionID);
    /*
     * Will not be used in the ApplicationClient
     */
    am_Error_e setVolume(const am_sinkID_t sinkID, const am_mainVolume_t volume);
    am_Error_e volumeStep(const am_sinkID_t sinkID, const int16_t volumeStep);
    am_Error_e setSinkMuteState(const am_sinkID_t sinkID, const am_MuteState_e muteState);
    am_Error_e setMainSinkSoundProperty(const am_MainSoundProperty_s& soundProperty, const am_sinkID_t sinkID);
    am_Error_e setMainSourceSoundProperty(const am_MainSoundProperty_s& soundProperty, const am_sourceID_t sourceID);
    am_Error_e setSystemProperty(const am_SystemProperty_s& property);
    am_Error_e getListMainConnections(std::vector<am_MainConnectionType_s>& listConnections);
    am_Error_e getListMainSinks(std::vector<am_SinkType_s>& listMainSinks);
    am_Error_e getListMainSources(std::vector<am_SourceType_s>& listMainSources);
    am_Error_e getListMainSinkSoundProperties(const am_sinkID_t sinkID,
                                              std::vector<am_MainSoundProperty_s>& listSoundProperties);
    am_Error_e getListMainSourceSoundProperties(const am_sourceID_t sourceID,
                                                std::vector<am_MainSoundProperty_s>& listSourceProperties);
    am_Error_e getListSourceClasses(std::vector<am_SourceClass_s>& listSourceClasses);
    am_Error_e getListSinkClasses(std::vector<am_SinkClass_s>& listSinkClasses);
    am_Error_e getListSystemProperties(std::vector<am_SystemProperty_s>& listSystemProperties);
    am_Error_e getTimingInformation(const am_mainConnectionID_t mainConnectionID, am_timeSync_t& delay);
    void confirmCommandReady(const uint16_t handle, const am_Error_e error);
    void confirmCommandRundown(const uint16_t handle, const am_Error_e error);
    am_Error_e getListMainSinkNotificationConfigurations(
            const am_sinkID_t sinkID, std::vector<am_NotificationConfiguration_s>& listMainNotificationConfigurations);
    am_Error_e getListMainSourceNotificationConfigurations(
            const am_sourceID_t sourceID, std::vector<am_NotificationConfiguration_s>& listMainNotificationConfigurations);
    am_Error_e setMainSinkNotificationConfiguration(const am_sinkID_t sinkID,
                                                    const am_NotificationConfiguration_s& mainNotificationConfiguration);
    am_Error_e setMainSourceNotificationConfiguration(const am_sourceID_t sourceID,
                                                      const am_NotificationConfiguration_s& mainNotificationConfiguration);
    am_Error_e getVolume(const am_sinkID_t sinkID, am_mainVolume_t& mainVolume);

    /* Sender callbacks */

    /*
     * To prevent the race condition on creation phase (the problem is that
     * IAmCommandClient is launching the workerThread listening on IPC during
     * ctor, and there, we are not assured that virtual function table is complete;
     * so, it can happen that a message arrives to IPC before the vtable is
     * finalized, leading to a crash) all the following functions will be declared
     * virtual and not pure virtual, providing a stub implementation here. But
     * then, we'll use C++11 pure virtual overrider in the upper layer interfaces
     * (IAm*Client) to force the final user to implement such methods.
     */
public:
    virtual void setCommandReady(const uint16_t handle);
    virtual void setCommandRundown(const uint16_t handle);
    virtual void cbNewMainConnection(const am_MainConnectionType_s& mainConnectionType);
    virtual void cbRemovedMainConnection(const am_mainConnectionID_t mainConnection);
    virtual void cbNewSink(const am_SinkType_s& sink);
    virtual void cbRemovedSink(const am_sinkID_t sinkID);
    virtual void cbNewSource(const am_SourceType_s& source);
    virtual void cbRemovedSource(const am_sourceID_t source);
    virtual void cbNumberOfSinkClassesChanged();
    virtual void cbNumberOfSourceClassesChanged();
    virtual void cbMainConnectionStateChanged(const am_mainConnectionID_t connectionID,
                                              const am_ConnectionState_e connectionState);
    virtual void cbMainSinkSoundPropertyChanged(const am_sinkID_t sinkID, const am_MainSoundProperty_s& soundProperty);
    virtual void cbMainSourceSoundPropertyChanged(const am_sourceID_t sourceID,
                                                  const am_MainSoundProperty_s& soundProperty);
    virtual void cbSinkAvailabilityChanged(const am_sinkID_t sinkID, const am_Availability_s& availability);
    virtual void cbSourceAvailabilityChanged(const am_sourceID_t sourceID, const am_Availability_s& availability);
    virtual void cbVolumeChanged(const am_sinkID_t sinkID, const am_mainVolume_t volume);
    virtual void cbSinkMuteStateChanged(const am_sinkID_t sinkID, const am_MuteState_e muteState);
    virtual void cbSystemPropertyChanged(const am_SystemProperty_s& systemProperty);
    virtual void cbTimingInformationChanged(const am_mainConnectionID_t mainConnectionID, const am_timeSync_t time);
    virtual void cbSinkUpdated(const am_sinkID_t sinkID, const am_sinkClass_t sinkClassID,
                               const std::vector<am_MainSoundProperty_s>& listMainSoundProperties);
    virtual void cbSourceUpdated(const am_sourceID_t sourceID, const am_sourceClass_t sourceClassID,
                                 const std::vector<am_MainSoundProperty_s>& listMainSoundProperties);
    virtual void cbSinkNotification(const am_sinkID_t sinkID, const am_NotificationPayload_s& notification);
    virtual void cbSourceNotification(const am_sourceID_t sourceID, const am_NotificationPayload_s& notification);
    virtual void cbMainSinkNotificationConfigurationChanged(
            const am_sinkID_t sinkID, const am_NotificationConfiguration_s& mainNotificationConfiguration);
    virtual void cbMainSourceNotificationConfigurationChanged(
            const am_sourceID_t sourceID, const am_NotificationConfiguration_s& mainNotificationConfiguration);

public:
    void setIAmCommandReceive(IAmCommandReceive *commandReceive)
    {
        mpIAmCommandReceive = commandReceive;
    }
    IAmCommandReceive *getIAmCommandReceive()
    {
        return mpIAmCommandReceive;
    }

    void setCAmSerializer(V2::CAmSerializer *serializer)
    {
        mpSerializer = serializer;
    }
    V2::CAmSerializer *getCAmSerializer()
    {
        return mpSerializer;
    }

private:
    pthread_mutex_t    mMutex;
    IAmCommandReceive  *mpIAmCommandReceive;
    V2::CAmSerializer      *mpSerializer;
};

}
#endif /* _I_DBUS_COMMAND_CLIENT_H_ */
